//methods for Verlet class. The constructors set the data members of the Verlet class. The initialize() function prepares the system to be time stepped by the integrate() function which contains the main molecular dynamics step loop.
#include "Verlet/Verlet.h"
#include "Verlet/Force.h"
#include <iostream.h>
#include <stdlib.h>
#include <math.h>


Verlet::Verlet() {
    cout << "enter the initial temperature.\n initial temperature = ";
    cin >> m_temperature;
    cout << "enter the number of particles.\n number of particles = ";
    cin >> m_numberOfParticles;
    cout << "enter the density.\n density = ";
    cin >> m_density;
    cout << "enter the time step.\n time step = ";
    cin >> m_timeStep;
    cout << "enter the number of time steps.\n number of time steps = ";
    cin >> m_numberOfTimeSteps;
}

Verlet::Verlet( const double temperature, const int numberOfParticles, const double density, const double timeStep, const int numberOfTimeSteps ) {
    m_temperature = temperature;
    m_numberOfParticles = numberOfParticles;
    m_density = density;
    m_timeStep = timeStep;
    m_numberOfTimeSteps = numberOfTimeSteps;
}

Verlet::~Verlet () {}

void Verlet::Initialize () {
  
  //edge length of entire volume of particles.
  m_bigEdge = pow ( ((double)(m_numberOfParticles))/m_density, 1./3. );

  //edge length of the little cube.
  m_littleEdge = pow ( m_density, -1./3. );

  //cube root of the number of particles in the volume.
  int cubeRootN = (int)(rint(pow ( ((double)(m_numberOfParticles)), 1./3. )));

  //center of mass velocities.
  double cmVx = 0.0;
  double cmVy = 0.0;
  double cmVz = 0.0;
  
  //sum of velocities squared
  double sumVsquared = 0.0; 

  for ( int zIter = 0; zIter<cubeRootN; zIter++ ) {
     for ( int yIter = 0; yIter<cubeRootN; yIter++ ) {
        for ( int xIter = 0; xIter<cubeRootN; xIter++ ) {

	   // initialize particle positions and velocities.
	   Particle aParticle( -m_bigEdge/2. + m_littleEdge/2. + ((double)(xIter))*m_littleEdge, -m_bigEdge/2. + m_littleEdge/2. + ((double)(yIter))*m_littleEdge, -m_bigEdge/2. + m_littleEdge/2. + ((double)(zIter))*m_littleEdge, ((double)(rand()))/((double)(RAND_MAX))-0.5, ((double)(rand()))/((double)(RAND_MAX))-0.5, ((double)(rand()))/((double)(RAND_MAX))-0.5 );
	   m_particleVector.push_back( aParticle );

	   cmVx += aParticle.Getvx();
	   cmVy += aParticle.Getvy();
	   cmVz += aParticle.Getvz();

	}
     }
  }

  cmVx /= m_numberOfParticles;
  cmVy /= m_numberOfParticles;
  cmVz /= m_numberOfParticles;

  // for debugging purposes.
  //cout << "big edge = " << m_bigEdge << endl << "little edge = " << m_littleEdge << endl << "cubeRootN = " << cubeRootN << endl << "CM vx = " << cmVx << endl << "CM vy = " << cmVy << endl << "CM vz = " << cmVz << endl;

  for ( vector<Particle>::iterator iter = m_particleVector.begin(); iter != m_particleVector.end(); iter++ ) {
     
    iter->Setvx( (iter->Getvx()-cmVx) );
    iter->Setvy( (iter->Getvy()-cmVy) );
    iter->Setvz( (iter->Getvz()-cmVz) );

    sumVsquared += iter->Getvx()*iter->Getvx() + iter->Getvy()*iter->Getvy() + iter->Getvz()*iter->Getvz();
  
  }

  //scale factor for velocities to ensure that total kinetic energy is zero.
  double scaleFactor = sqrt ( 3*((double)(m_numberOfParticles))*m_temperature/sumVsquared );

  for ( vector<Particle>::iterator iter = m_particleVector.begin(); iter != m_particleVector.end(); iter++ ) {
    
    iter->Setvx( iter->Getvx()*scaleFactor );
    iter->Setvy( iter->Getvy()*scaleFactor );
    iter->Setvz( iter->Getvz()*scaleFactor );

    iter->SetPrevx( iter->Getx() - iter->Getvx()*m_timeStep );
    iter->SetPrevy( iter->Gety() - iter->Getvy()*m_timeStep );
    iter->SetPrevz( iter->Getz() - iter->Getvz()*m_timeStep );

    //for debugging the particles initial conditions.
    //cout << "x = " << iter->Getx() << ", y = " << iter->Gety() << ", z = " << iter->Getz() << ", vx = " << iter->Getvx() << ", vy = " << iter->Getvy() << ", vz = " << iter->Getvz() << endl;

  } 
  
  //This block is used to debug the conservation of the intialization.
  /*
  cmVx = 0.0;
  cmVy = 0.0;
  cmVz = 0.0;
  sumVsquared = 0.0;
  
  for ( vector<Particle>::iterator iter = m_particleVector.begin(); iter != m_particleVector.end(); iter++ ) {
    
    cmVx += iter->Getvx();
    cmVy += iter->Getvy();
    cmVz += iter->Getvz();

    sumVsquared += (iter->Getvx())*(iter->Getvx()) + (iter->Getvy())*(iter->Getvy()) + (iter->Getvz())*(iter->Getvz());

  }

  cout << "cm vx = " << cmVx << endl << "cm vy = " << cmVy << endl << "cm vz = " << cmVz << endl << "sum v squared = " << sumVsquared << endl;
  */

}


void Verlet::Integrate ( vector<FunctionPtr> externalFunctions ) {
   
   //instantiate force function object.
   Force theForce;
 
   //local variables.
   double separationx = 0.0;
   double separationy = 0.0;
   double separationz = 0.0;
   double separation = 0.0;
   double xForce = 0.0;
   double yForce = 0.0;
   double zForce = 0.0;

   //time stepping loop.
   for ( int timeStep = 0; timeStep<m_numberOfTimeSteps; timeStep++ ) {

      //set forces to zero for every particle at each time step.
      for ( vector<Particle>::iterator particle = m_particleVector.begin(); particle != m_particleVector.end(); particle++ ) {
	 particle->zeroForce();
      } 

      //force calculation loop.
      for ( vector<Particle>::iterator i = m_particleVector.begin(); i != m_particleVector.end()-1; i++ ) {
         for ( vector<Particle>::iterator j = i+1; j != m_particleVector.end(); j++ ) {
	  
	   //calculate interparticle spacings using periodic boundary condition.
	   separationx = (i->Getx()-j->Getx());
	   separationx -= m_bigEdge*rint(separationx/m_bigEdge);
  	   separationy = (i->Gety()-j->Gety());
	   separationy -= m_bigEdge*rint(separationy/m_bigEdge);
	   separationz = (i->Getz()-j->Getz());
	   separationz -= m_bigEdge*rint(separationz/m_bigEdge);

	   separation = sqrt( (separationx*separationx) + (separationy*separationy) + (separationz*separationz) );


	   //for debugging the separation calculation.
	   /*
	     cout << "separation x = " << separationx << ", separation y = " << separationy << ", separation z = " << separationz << ", full sep = " << separation << endl;
	*/
	
	   xForce = theForce( separationx, separation );
	   yForce = theForce( separationy, separation );
	   zForce = theForce( separationz, separation );

	   i->updateXforce( xForce );
	   j->updateXforce( -xForce );
	   i->updateYforce( yForce );
	   j->updateYforce( -yForce );
	   i->updateZforce( zForce );
	   j->updateZforce( -zForce );

	 }
      }
   
      // used to debug force calculation
      /*
      for ( vector<Particle>::iterator particle = m_particleVector.begin(); particle != m_particleVector.end(); particle++ ) {
        cout << "x force = " << particle->GetXforce() << ", y force = " << particle->GetYforce() << ", z force = " << particle->GetZforce() << endl;
      }
      */

      //each particle integrates its own position and velocity individually.
      for ( vector<Particle>::iterator particle = m_particleVector.begin(); particle != m_particleVector.end(); particle++ ) {
	 particle->integrate ( m_timeStep );
      }

      //execute external functions.
      for ( vector<FunctionPtr>::iterator functionIter = externalFunctions.begin(); functionIter != externalFunctions.end(); functionIter++ ) {
	 functionIter->m_ptr2function( &m_temperature, &m_numberOfParticles, &m_density, &m_timeStep, &m_particleVector, &m_bigEdge, &m_littleEdge, &m_numberOfTimeSteps, &timeStep );
      }

   }

}
