//Header file for the RadialDistribution() function used to calculate the radial distribution function g(r) of the system of particles. This file implements the algorithm presented on page 86 of "Understanding Molecular Simulation" by Frenkel and Smit.
#ifndef RADIALDISTRIBUTION_H
#define RADIALDISTRIBUTION_H

//select the number of bins for the g(r) histogram here.
#define NUMBER_OF_BINS 256
//select frequency to calculate g(r).
#define GOFR_FREQUENCY 8

#include <deque.h>
#include <fstream.h>
#include <math.h>
#include "Verlet/Particle.h"

void RadialDistribution( double * temperature, int * numberOfParticles, double * density, double * timeStep, vector<Particle> * particleVector, double * bigEdge, double * littleEdge, int * numberOfTimeSteps, int * thisTimeStep ) {

  //Remove output file "RadialDistribution.dat" if it exists.
  if ( *thisTimeStep == 0 ) {
    ofstream gOfRFile( "output/RadialDistribution.dat", ios::out );
    gOfRFile.close();
  }

  //only calculate the radial distribution function at the desired frequency.
  if ( *thisTimeStep % GOFR_FREQUENCY == 0 ) {

      //vector<double> not compatible with some versions of libc.so, using deque appears to eliminate the seg faults.
     deque<double> gOfR;

     const double deltaR = *bigEdge/(2.*((double)(NUMBER_OF_BINS)));
     const double pi = 4.*atan(1.0);

  
     for ( int i=0; i<NUMBER_OF_BINS; i++ ) {
       gOfR.push_back(0.0); 
     }
  
     double separationx = 0.0;
     double separationy = 0.0;
     double separationz = 0.0;
     double separation = 0.0;
     int binNumber = 0;

  
     //loop over all paris of particles.
     for ( vector<Particle>::iterator i = particleVector->begin(); i != particleVector->end()-1; i++ ) {
        for ( vector<Particle>::iterator j = i+1; j != particleVector->end(); j++ ) {

       
	  separationx = (i->Getx()-j->Getx());
	  separationx -= (*bigEdge)*rint(separationx/(*bigEdge));
	  separationy = (i->Gety()-j->Gety());
	  separationy -= (*bigEdge)*rint(separationy/(*bigEdge));
	  separationz = (i->Getz()-j->Getz());
	  separationz -= (*bigEdge)*rint(separationz/(*bigEdge));

	  separation = sqrt((separationx*separationx)+(separationy*separationy)+(separationz*separationz));

	  //spherically symmetric. must be within half box-length.
	  if ( separation < (*bigEdge/2) ) {
	     binNumber = ((int)(rint(separation/deltaR)));
	     gOfR[binNumber] += 2.0;
	  }
       
	}
     }

     //now calculate g(r)
     double binVolume = 0.0;
     double idealNumber = 0.0;

     ofstream gOfRfile( "output/RadialDistribution.dat", ios::app );

     for ( int i=0; i<NUMBER_OF_BINS; i++ ) {

        binVolume = (4./3.)*pi*( ((double)((i+1)*(i+1)*(i+1)-i*i*i)) )*(deltaR*deltaR*deltaR);
	idealNumber = binVolume*(*density);
	gOfR[i] /= idealNumber*((double)(*numberOfParticles));
	gOfRfile << gOfR[i] << "  ";

     }
  
     gOfRfile << endl << endl;
  
     gOfRfile.close();
  
  }

}

#endif

